package com.xc.apex.nre.nrep8posdemo.edgeos

import com.xc.apex.nre.nrep8posdemo.edgeos.methods.AddGoodsMethod
import com.xc.apex.nre.nrep8posdemo.edgeos.methods.BatchAddGoodsMethod
import com.xc.apex.nre.nrep8posdemo.edgeos.methods.ClearGoodsMethod
import com.xc.apex.nre.nrep8posdemo.edgeos.methods.GetGoodsListMethod
import com.xc.apex.nre.nrep8posdemo.edgeos.methods.GetSelectedGoods
import com.xc.apex.nre.nrep8posdemo.edgeos.methods.GoHomeMethod
import com.xc.apex.nre.nrep8posdemo.edgeos.methods.RemoveGoodsMethod
import com.xc.apex.nre.nrep8posdemo.edgeos.methods.SubmitOrderMethod
import com.xctech.collaboration.spec.Implementation
import com.xctech.collaboration.spec.ServerCapabilities
import com.xctech.collaboration.spec.ToolServerCapability
import com.xctech.collaboration.tool.GeneratedToolService
import com.xctech.collaboration.tool.annotation.Param
import com.xctech.collaboration.tool.annotation.ToolMethod
import com.xctech.collaboration.tool.annotation.ToolService
import com.xctech.collaboration.tool.ToolCallCallback
import com.xctech.collaboration.tool.successText
import org.json.JSONObject
import java.util.HashMap

/**
 * EdgeOS Tool 服务类
 * 作为包装层，调用原有方法类的 execute() 方法
 */
@ToolService(
    name = "MEOW PAY",
    version = "1.0.0",
    description = "MEOW PAY点餐服务"
)
open class EdgeOSToolService : GeneratedToolService(){

    /**
     * 添加商品到购物车
     */
    @ToolMethod(
        name = "addGoods",
        description = "添加商品到购物车 / Add product to shopping cart."
    )
    fun addGoods(
        @Param(name = "id", description = "商品id / Goods ID", required = false) id: String?,
        @Param(name = "name", description = "商品名称，支持模糊匹配 / Goods name, supports fuzzy matching", required = false) name: String?,
        @Param(name = "quantity", description = "数量 / quantity", required = false) quantity: String?,
        callback: ToolCallCallback
    ) {
        val method = EdgeOSHelper.getMethod("addGoods") as? AddGoodsMethod
        if (method == null) {
            callback.onError("方法未找到")
            return
        }

        // 转换为 Map 格式
        val params = HashMap<String, Any?>()
        id?.let { params["id"] = it }
        name?.let { params["name"] = it }
        params["quantity"] = quantity ?: "1"

        // 调用原有方法
        val result = method.execute(params)

        // 转换返回值
        convertJsonResultToCallback(result, callback)
    }

    /**
     * 清空购物车
     */
    @ToolMethod(
        name = "clearGoods",
        description = "调用该方法清空购物车内所有商品 / Call this method to clear all items in the shopping cart"
    )
    fun clearGoods(callback: ToolCallCallback) {
        val method = EdgeOSHelper.getMethod("clearGoods") as? ClearGoodsMethod
        if (method == null) {
            callback.onError("方法未找到")
            return
        }

        val result = method.execute(emptyMap<String, Any?>())
        convertJsonResultToCallback(result, callback)
    }

    /**
     * 提交订单
     */
    @ToolMethod(
        name = "submitOrder",
        description = "提交当前购物车内的商品并生成订单 / Submit the goods in the current shopping cart and generate an order"
    )
    fun submitOrder(callback: ToolCallCallback) {
        val method = EdgeOSHelper.getMethod("submitOrder") as? SubmitOrderMethod
        if (method == null) {
            callback.onError("方法未找到")
            return
        }

        val result = method.execute(emptyMap<String, Any?>())
        convertJsonResultToCallback(result, callback)
    }

    /**
     * 返回首页
     */
    @ToolMethod(
        name = "goHome",
        description = "返回点餐页面 / Return to the ordering page"
    )
    fun goHome(callback: ToolCallCallback) {
        val method = EdgeOSHelper.getMethod("goHome") as? GoHomeMethod
        if (method == null) {
            callback.onError("方法未找到")
            return
        }

        val result = method.execute(emptyMap<String, Any?>())
        convertJsonResultToCallback(result, callback)
    }

    /**
     * 移除商品
     */
    @ToolMethod(
        name = "removeGoods",
        description = "将商品从购物车移除 / Remove product from shopping cart"
    )
    fun removeGoods(
        @Param(name = "id", description = "商品id / Goods ID", required = false) id: String?,
        @Param(name = "name", description = "商品名称，支持模糊匹配 / Goods name, supports fuzzy matching", required = false) name: String?,
        @Param(name = "quantity", description = "移除数量 / quantity", required = false) quantity: String?,
        callback: ToolCallCallback
    ) {
        val method = EdgeOSHelper.getMethod("removeGoods") as? RemoveGoodsMethod
        if (method == null) {
            callback.onError("方法未找到")
            return
        }

        // 转换为 Map 格式
        val params = HashMap<String, Any?>()
        id?.let { params["id"] = it }
        name?.let { params["name"] = it }
        params["quantity"] = quantity ?: "1"

        val result = method.execute(params)
        convertJsonResultToCallback(result, callback)
    }

    /**
     * 获取商品列表
     */
    @ToolMethod(
        name = "getGoods",
        description = "获取商品列表"
    )
    fun getGoods(callback: ToolCallCallback) {
        val method = EdgeOSHelper.getMethod("getGoods") as? GetGoodsListMethod
        if (method == null) {
            callback.onError("方法未找到")
            return
        }

        val result = method.execute(emptyMap<String, Any?>())
        // GetGoodsListMethod 返回 String
        callback.successText(result)
    }

    /**
     * 批量添加商品
     */
    @ToolMethod(
        name = "batchAddGoods",
        description = "批量添加商品"
    )
    fun batchAddGoods(
        @Param(name = "list", description = "[{\"id\":int,\"quantity\":int},{...}]的格式批量添加商品") list: String,
        callback: ToolCallCallback
    ) {
        val method = EdgeOSHelper.getMethod("batchAddGoods") as? BatchAddGoodsMethod
        if (method == null) {
            callback.onError("方法未找到")
            return
        }

        // 转换为 Map 格式
        val params = HashMap<String, Any?>()
        params["list"] = list

        val result = method.execute(params)
        convertJsonResultToCallback(result, callback)
    }

    /**
     * 获取已选商品列表
     */
    @ToolMethod(
        name = "getSelectedGoods",
        description = "获取当前已点选的菜品列表"
    )
    fun getSelectedGoods(callback: ToolCallCallback) {
        val method = EdgeOSHelper.getMethod("getSelectedGoods") as? GetSelectedGoods
        if (method == null) {
            callback.onError("方法未找到")
            return
        }

        val result = method.execute(emptyMap<String, Any?>())
        // GetSelectedGoods 返回 String
        callback.successText(result)
    }

    /**
     * 将 JSONObject 返回值转换为 callback 调用
     */
    private fun convertJsonResultToCallback(result: Any?, callback: ToolCallCallback) {
        when (result) {
            is JSONObject -> {
                val success = result.optBoolean("success", false)
                val message = result.optString("message", "")
                if (success) {
                    callback.successText(message)
                } else {
                    callback.onError(message)
                }
            }
            is String -> {
                callback.successText(result)
            }
            else -> {
                callback.successText(result?.toString() ?: "")
            }
        }
    }

    override fun getToolServerInfo(): Implementation {
        return Implementation(name = "MEOW PAY", version = "1.0.0")
    }

    override fun getServerCapabilities(): ServerCapabilities {
        return ServerCapabilities(
            tools = ToolServerCapability(
                listChanged = true
            )
        )
    }
}

