package com.xc.apex.nre.nrep8posdemo.view

import android.Manifest
import android.animation.Animator
import android.animation.AnimatorListenerAdapter
import android.animation.AnimatorSet
import android.animation.ObjectAnimator
import android.animation.PropertyValuesHolder
import android.animation.ValueAnimator
import android.content.pm.PackageManager
import android.os.Bundle
import android.os.Handler
import android.os.Looper
import android.util.Log
import android.view.View
import android.view.animation.AccelerateDecelerateInterpolator
import android.view.animation.LinearInterpolator

import androidx.core.app.ActivityCompat
import androidx.core.content.ContextCompat
import androidx.databinding.DataBindingUtil
import androidx.fragment.app.Fragment

import com.xc.apex.nre.nrep8posdemo.BaseApplicationV2
import com.xc.apex.nre.nrep8posdemo.R
import com.xc.apex.nre.nrep8posdemo.databinding.ActivityMainTabBinding
import com.xc.apex.nre.nrep8posdemo.edgeos.EdgeOSService
import com.xc.apex.nre.nrep8posdemo.utils.CommonUtils
import com.xc.apex.nre.nrep8posdemo.utils.ToastUtil
import com.xc.apex.nre.nrep8posdemo.view.base.BaseActivity
import com.xc.apex.nre.nrep8posdemo.view.fragment.ChargeFragment
import com.xc.apex.nre.nrep8posdemo.view.fragment.ItemsFragment
import com.xc.apex.nre.nrep8posdemo.view.fragment.ReceiptsFragment
import com.xc.apex.nre.nrep8posdemo.view.fragment.SalesFragment
import com.xc.apex.nre.nrep8posdemo.view.fragment.SettingsFragment
import com.xc.apex.nre.nrep8posdemo.view.widget.CustomTabView
import com.xc.apex.nre.nrep8posdemo.view.widget.VoiceWaveView

import com.xctech.collaboration.sdk.CollaborationSDK
import com.xctech.collaboration.core.TaskCallback
import com.xctech.collaboration.core.TaskError
import com.xctech.collaboration.core.TaskConfig
import com.xctech.collaboration.core.CommandResult

class MainTabActivity : BaseActivity(), View.OnClickListener {
    companion object {
        private const val TAG = "MainTabActivity"
        private const val REQUEST_CODE_AUDIO_PERMISSION = 1001
    }

    private lateinit var binding: ActivityMainTabBinding
    private lateinit var salesFragment: SalesFragment
    private lateinit var receiptsFragment: ReceiptsFragment
    private lateinit var itemsFragment: ItemsFragment
    private lateinit var settingsFragment: SettingsFragment
    private var chargeFragment: ChargeFragment? = null
    private var currentFragment: Fragment? = null

    // AI录音状态
    private var isRecording = false
    private var isAnalyzing = false
    private var transcript = ""

    // 声波动画View
    private var voiceWaveView: VoiceWaveView? = null

    // 脉冲动画
    private var pulseAnimator: AnimatorSet? = null

    private val mHandler = Handler(Looper.getMainLooper())

    // 当前任务ID（用于取消任务）
    private var currentTaskId: String? = null

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        binding = DataBindingUtil.setContentView(this, R.layout.activity_main_tab)
        binding.presenter = this

        // 初始化DataBinding变量
        binding.isRecording = false
        binding.isAnalyzing = false
        binding.transcript = ""

        salesFragment = SalesFragment()
        receiptsFragment = ReceiptsFragment()
        itemsFragment = ItemsFragment()
        settingsFragment = SettingsFragment()

        binding.tabSales.setTabSelected()
        supportFragmentManager.beginTransaction()
            .add(R.id.fragment_container, salesFragment)
            .commit()
        currentFragment = salesFragment

        // 初始化声波动画View
        initVoiceWaveView()

        val hasCameraPermission = CommonUtils.checkPermissionAndCamera(this)
        if (!hasCameraPermission) {
            ActivityCompat.requestPermissions(
                this,
                arrayOf(Manifest.permission.CAMERA),
                CommonUtils.REQUEST_CODE_QRCODE_PERMISSIONS
            )
        }

        // 注册EdgeOS协同
        EdgeOSService.init(this, salesFragment)
    }

    private fun initVoiceWaveView() {
        voiceWaveView = VoiceWaveView(this)
        binding.voiceWaveContainer.addView(voiceWaveView)
    }

    override fun onClick(view: View) {
        when (view.id) {
            R.id.tab_sales -> switchTabFragment(binding.tabSales, salesFragment)
            R.id.tab_receipts -> switchTabFragment(binding.tabReceipts, receiptsFragment)
            R.id.tab_items -> switchTabFragment(binding.tabItems, itemsFragment)
            R.id.tab_setting -> switchTabFragment(binding.tabSetting, settingsFragment)
        }
    }

    /**
     * AI语音按钮点击事件
     */
    fun onAiVoiceClick(view: View) {
        if (!isRecording && !isAnalyzing) {
            // 检查录音权限
            startRecording()
        } else if (isRecording) {
            stopRecording()
        }
    }

    /**
     * 开始录音
     */
    private fun startRecording() {
        isRecording = true
        isAnalyzing = false
        transcript =  getString(R.string.ai_voice_listening_hint)

        updateBindingState()

        // 切换按钮样式为录音态（白色背景）
        binding.layoutBtnContent.setBackgroundResource(R.drawable.bg_ai_button_recording)

        // 按钮放大动画
        animateButtonScale(1.25f)

        // 显示遮罩层动画
        showMaskOverlay()

        // 开始脉冲动画
        startPulseAnimation()

        // 开始声波动画
        voiceWaveView?.startAnimation()

        // 使用新的 SDK 触发 Michael Agent 模块
        sendCommandToMichael(BaseApplicationV2.COMMAND_START_MINCHALE)
    }

    /**
     * 停止录音并开始分析
     */
    private fun stopRecording() {
        isRecording = false
        isAnalyzing = true
        transcript = "AI 正在分析订单..."

        updateBindingState()

        // 切换按钮样式为分析态（深色背景）
        binding.layoutBtnContent.setBackgroundResource(R.drawable.bg_ai_button_analyzing)

        // 按钮恢复正常大小
        animateButtonScale(1f)

        // 停止脉冲动画
        stopPulseAnimation()

        // 停止声波动画
        voiceWaveView?.stopAnimation()

        // 隐藏遮罩层
        hideMaskOverlay()

        // 使用新的 SDK 触发 Michael Agent 模块停止命令
//        sendCommandToMichael(BaseApplicationV2.COMMAND_STOP_MINCHALE)
    }

    /**
     * 模拟AI分析过程
     */
    private fun simulateAnalysis() {
        mHandler.postDelayed({
            isAnalyzing = false
            transcript = "已为您添加：牛肉面、冰可乐"

            updateBindingState()

            // 恢复按钮默认样式
            binding.layoutBtnContent.setBackgroundResource(R.drawable.bg_ai_button_normal)

//            // 显示气泡提示
//            showBubbleTip(transcript)
//
//            // 3秒后隐藏气泡
//            mHandler.postDelayed({
//                hideBubbleTip()
//                transcript = ""
//                binding.transcript = ""
//            }, 3000)
//        }, 2000)
        }, 2000)
    }

    /**
     * 发送命令到 Michael Agent 模块
     */
    private fun sendCommandToMichael(commandText: String) {
        val sdk = BaseApplicationV2.getCollaborationSDK()
        if (sdk == null) {
            Log.e(TAG, "SDK 未初始化")
            mHandler.post {
                ToastUtil.showToast(this, "SDK 未初始化")
            }
            return
        }

        // 发送命令
        currentTaskId = sdk.command(
            text = commandText,
            config = TaskConfig(timeout = 60_000L), // 60秒超时
            callback = object : TaskCallback<CommandResult> {
                override fun onStart(taskId: String) {
                    Log.d(TAG, "任务开始: $taskId")
                    mHandler.post {
                        // 任务开始时的UI更新已在startRecording/stopRecording中处理
                    }
                }

                override fun onProgress(taskId: String, chunk: CommandResult) {
                    Log.d(TAG, "任务进度更新: $taskId")
                    mHandler.post {
                        // 处理流式进度更新
                        val progressText = chunk.data ?: chunk.message
                        if (!progressText.isNullOrEmpty()) {
                            transcript = progressText
                            updateBindingState()
                        }
                    }
                }

                override fun onComplete(taskId: String, result: CommandResult) {
                    Log.d(TAG, "任务完成: $taskId, 结果: $result")
                    mHandler.post {
                        if (isAnalyzing) {
                            // 分析完成
                            isAnalyzing = false
                            val resultText = result.data ?: result.message

                            if (!resultText.isNullOrEmpty()) {
                                transcript = resultText
                            } else {
                                transcript = "分析完成"
                            }
                            updateBindingState()

                            // 恢复按钮默认样式
                            binding.layoutBtnContent.setBackgroundResource(R.drawable.bg_ai_button_normal)

                            // 显示气泡提示
                            showBubbleTip(transcript)

                            // 3秒后隐藏气泡
                            mHandler.postDelayed({
                                hideBubbleTip()
                                transcript = ""
                                binding.transcript = ""
                            }, 3000)
                        }
                    }
                }

                override fun onError(taskId: String, error: TaskError) {
                    Log.e(TAG, "任务错误: $taskId, 错误: ${error.message}")
                    mHandler.post {
                        isRecording = false
                        isAnalyzing = false
                        transcript = "错误: ${error.message}"
                        updateBindingState()
                        binding.layoutBtnContent.setBackgroundResource(R.drawable.bg_ai_button_normal)
                        ToastUtil.showToast(this@MainTabActivity, "错误: ${error.message}")
                    }
                }

                override fun onTimeout(taskId: String) {
                    Log.e(TAG, "任务超时: $taskId")
                    mHandler.post {
                        isRecording = false
                        isAnalyzing = false
                        transcript = "请求超时，请重试"
                        updateBindingState()
                        binding.layoutBtnContent.setBackgroundResource(R.drawable.bg_ai_button_normal)
                        ToastUtil.showToast(this@MainTabActivity, "请求超时，请重试")
                    }
                }

                override fun onCancelled(taskId: String) {
                    Log.d(TAG, "任务已取消: $taskId")
                    mHandler.post {
                        isRecording = false
                        isAnalyzing = false
                        transcript = "已取消"
                        updateBindingState()
                        binding.layoutBtnContent.setBackgroundResource(R.drawable.bg_ai_button_normal)
                    }
                }
            }
        )
    }

    /**
     * 更新DataBinding状态
     */
    private fun updateBindingState() {
        binding.isRecording = isRecording
        binding.isAnalyzing = isAnalyzing
        binding.transcript = transcript
    }

    /**
     * 按钮缩放动画
     */
    private fun animateButtonScale(targetScale: Float) {
        val scaleX = ObjectAnimator.ofFloat(binding.btnAiVoice, "scaleX", targetScale)
        val scaleY = ObjectAnimator.ofFloat(binding.btnAiVoice, "scaleY", targetScale)

        val animatorSet = AnimatorSet()
        animatorSet.playTogether(scaleX, scaleY)
        animatorSet.duration = 300
        animatorSet.interpolator = AccelerateDecelerateInterpolator()
        animatorSet.start()
    }

    /**
     * 显示遮罩层
     */
    private fun showMaskOverlay() {
        binding.maskOverlay.visibility = View.VISIBLE
        val fadeIn = ObjectAnimator.ofFloat(binding.maskOverlay, "alpha", 0f, 1f)
        fadeIn.duration = 500
        fadeIn.start()
    }

    /**
     * 隐藏遮罩层
     */
    private fun hideMaskOverlay() {
        val fadeOut = ObjectAnimator.ofFloat(binding.maskOverlay, "alpha", 1f, 0f)
        fadeOut.duration = 300
        fadeOut.addListener(object : AnimatorListenerAdapter() {
            override fun onAnimationEnd(animation: Animator) {
                binding.maskOverlay.visibility = View.GONE
            }
        })
        fadeOut.start()
    }

    /**
     * 开始脉冲动画
     */
    private fun startPulseAnimation() {
        binding.pulseRing.visibility = View.VISIBLE
        binding.pulseRing.alpha = 0.3f

        val scaleX = PropertyValuesHolder.ofFloat("scaleX", 1f, 1.5f)
        val scaleY = PropertyValuesHolder.ofFloat("scaleY", 1f, 1.5f)
        val alpha = PropertyValuesHolder.ofFloat("alpha", 0.3f, 0f)

        val pulseAnim = ObjectAnimator.ofPropertyValuesHolder(
            binding.pulseRing, scaleX, scaleY, alpha
        )
        pulseAnim.duration = 2000
        pulseAnim.repeatCount = ValueAnimator.INFINITE
        pulseAnim.interpolator = LinearInterpolator()

        pulseAnimator = AnimatorSet()
        pulseAnimator?.play(pulseAnim)
        pulseAnimator?.start()
    }

    /**
     * 停止脉冲动画
     */
    private fun stopPulseAnimation() {
        pulseAnimator?.cancel()
        binding.pulseRing.visibility = View.GONE
    }

    /**
     * 显示气泡提示
     */
    private fun showBubbleTip(text: String) {
        binding.tvBubbleText.text = text
        binding.bubbleTip.visibility = View.VISIBLE
        binding.bubbleTip.alpha = 0f
        binding.bubbleTip.translationY = 20f

        val fadeIn = ObjectAnimator.ofFloat(binding.bubbleTip, "alpha", 0f, 1f)
        val slideUp = ObjectAnimator.ofFloat(binding.bubbleTip, "translationY", 20f, 0f)

        val animatorSet = AnimatorSet()
        animatorSet.playTogether(fadeIn, slideUp)
        animatorSet.duration = 300
        animatorSet.start()
    }

    /**
     * 隐藏气泡提示
     */
    private fun hideBubbleTip() {
        val fadeOut = ObjectAnimator.ofFloat(binding.bubbleTip, "alpha", 1f, 0f)
        fadeOut.duration = 200
        fadeOut.addListener(object : AnimatorListenerAdapter() {
            override fun onAnimationEnd(animation: Animator) {
                binding.bubbleTip.visibility = View.GONE
            }
        })
        fadeOut.start()
    }

    private fun switchTabFragment(view: CustomTabView, fragment: Fragment) {
        if (!view.isTabSelected) {
//            view.isTabSelected = true
            switchFragment(fragment)
        }
    }

    fun switchFragment(fragment: Fragment) {
        mHandler.post {
            if (currentFragment != fragment) {
                if (!fragment.isAdded) {
                    supportFragmentManager.beginTransaction()
                        .add(R.id.fragment_container, fragment)
                        .hide(currentFragment!!)
                        .commit()
                } else {
                    supportFragmentManager.beginTransaction()
                        .hide(currentFragment!!)
                        .show(fragment)
                        .commit()
                }
                currentFragment = fragment
            }
        }
    }

    fun showChargeFragment(args: Bundle) {
        try {
            chargeFragment = ChargeFragment().apply {
                arguments = args
            }
            supportFragmentManager.beginTransaction()
                .add(R.id.fragment_container, chargeFragment!!)
                .hide(currentFragment!!)
                .commitAllowingStateLoss()
            currentFragment = chargeFragment
        } catch (e: Exception) {
            e.printStackTrace()
            Log.e(TAG, "showChargeFragment error: ${e.message}")
        }
    }

    fun hideChargeFragment() {
        try {
            supportFragmentManager.beginTransaction()
                .hide(currentFragment!!)
                .show(salesFragment)
                .commitAllowingStateLoss()
            currentFragment = salesFragment
            chargeFragment = null
        } catch (e: Exception) {
            e.printStackTrace()
            Log.e(TAG, "hideChargeFragment error: ${e.message}")
        }
    }

    override fun onDestroy() {
        super.onDestroy()
        voiceWaveView?.stopAnimation()
        pulseAnimator?.cancel()
        mHandler.removeCallbacksAndMessages(null)
    }
}