package com.xc.apex.nre.nrep8posdemo.view.fragment;

import android.content.Context;

import android.os.Bundle;
import android.os.Handler;
import android.util.Log;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.view.inputmethod.InputMethodManager;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import androidx.databinding.DataBindingUtil;
import androidx.recyclerview.widget.LinearLayoutManager;
import androidx.recyclerview.widget.RecyclerView;

import com.bumptech.glide.Glide;
import com.bumptech.glide.load.engine.DiskCacheStrategy;
import com.xc.apex.nre.lib_payment.PaymentManager;
import com.xc.apex.nre.nrep8posdemo.BaseApplicationV2;
import com.xc.apex.nre.nrep8posdemo.R;

import com.xc.apex.nre.nrep8posdemo.databinding.FragmentChargeBinding;
import com.xc.apex.nre.nrep8posdemo.model.ClearOrderEvent;
import com.xc.apex.nre.nrep8posdemo.model.OrderItemBean;
import com.xc.apex.nre.nrep8posdemo.utils.CommonUtils;
import com.xc.apex.nre.nrep8posdemo.utils.ToastUtil;
import com.xc.apex.nre.nrep8posdemo.view.MainTabActivity;
import com.xc.apex.nre.nrep8posdemo.view.adapter.ChargeOrderAdapter;
import com.xc.apex.nre.nrep8posdemo.view.base.BaseFragment;
import com.xc.apex.nre.nrep8posdemo.view.widget.CustomPayWay;

import org.greenrobot.eventbus.EventBus;

import java.util.ArrayList;
import java.util.List;

import cn.bingoogolapple.qrcode.core.BarcodeType;
import cn.bingoogolapple.qrcode.core.QRCodeView;

public class ChargeFragment extends BaseFragment implements View.OnClickListener, PaymentManager.PaymentResultListener, QRCodeView.Delegate {
    private static final String TAG = "ChargeFragment";

    private FragmentChargeBinding binding;
    private List<OrderItemBean> orderItemList = new ArrayList<>();
    private ChargeOrderAdapter adapter;
    private String ticketTotalPrice = "0.0";

    private static final int PAY_CASH = 0;
    private static final int PAY_CARD = 1;
    private static final int PAY_QRCODE = 2;
    private int curPayWay = PAY_CASH;

    private boolean hasPayResult = false; // 是否有支付结果，无论成功和失败

    private boolean hasCameraPermission = false;
    private boolean isCameraOpened = false; // 只要开启，在界面被关闭的时候再关闭（快速切换回报错）

    private final Handler handler = new Handler();
    // 显示结果界面的时间，时间到了自动关闭页面
    private static final int DELAY_CLOSE_PAGE_TIME = 10 * 1000;
    private Runnable closeRunnable = new Runnable() {
        @Override
        public void run() {
            ((MainTabActivity) getActivity()).hideChargeFragment();
        }
    };

    @Nullable
    @Override
    public View onCreateView(@NonNull LayoutInflater inflater, @Nullable ViewGroup container, @Nullable Bundle savedInstanceState) {
        Log.e(TAG, "onCreateView+");
        binding = DataBindingUtil.inflate(inflater, R.layout.fragment_charge, container, false);
        binding.setPresenter(this);

        // 绑定扫码相机
        binding.scanView.setDelegate(this);
        hasCameraPermission = CommonUtils.checkPermissionAndCamera(getActivity());

        if (getArguments() != null) {
            ArrayList<OrderItemBean> orderDatas = getArguments().getParcelableArrayList("order_data_key");
            for (OrderItemBean item : orderDatas) {
                orderItemList.add(item);
            }

            ticketTotalPrice = getArguments().getString("order_total_val_key");
            Log.e(TAG, "[CHARGE] size = " + orderItemList.size() + " , ticketTotalPrice = " + ticketTotalPrice);

            if (orderItemList.size() > 0) {
                adapter = new ChargeOrderAdapter(getActivity(), orderItemList);
                RecyclerView.LayoutManager layoutManager = new LinearLayoutManager(getActivity(), LinearLayoutManager.VERTICAL, false);
                binding.rvOrder.setLayoutManager(layoutManager);
                binding.rvOrder.setAdapter(adapter);
            }

            binding.tvTotalVal.setText(requireContext().getString(R.string.money_sign) + ticketTotalPrice);
            binding.tvPayTotalVal.setText(requireContext().getString(R.string.money_sign) + ticketTotalPrice);
        }

        refreshPayTabState(binding.cashPay, PAY_CASH);
        refreshPayPage(true);

        Glide.with(this)
                .asGif()
                .load(R.drawable.gif_card)
                .diskCacheStrategy(DiskCacheStrategy.RESOURCE)
                .into(binding.ivOthersPay);

        return binding.getRoot();
    }

    @Override
    public void onDestroyView() {
        super.onDestroyView();
        Log.e(TAG, "onDestroyView+");
        release();
        binding.scanView.onDestroy(); // 销毁二维码扫描控件
    }

    @Override
    public void onHiddenChanged(boolean hidden) {
        super.onHiddenChanged(hidden);
        Log.e(TAG, "onHiddenChanged+ hasPayResult = " + hasPayResult);
        if (hasPayResult) {
            // 清空首頁訂單列表
            EventBus.getDefault().post(new ClearOrderEvent(true));
        }
        release();
        binding.scanView.onDestroy(); // 销毁二维码扫描控件
    }

    @Override
    public void onClick(View view) {
        int viewId = view.getId();
        if (viewId == R.id.iv_back) {
            ((MainTabActivity) getActivity()).hideChargeFragment();
        } else if (viewId == R.id.cash_pay) {
            changePayWay(binding.cashPay, PAY_CASH);

            // 关闭卡支付
            closeCardPaymentAccess();
        } else if (viewId == R.id.card_pay) {
            changePayWay(binding.cardPay, PAY_CARD);

            // 开启卡支付
            openCardPaymentAccess();
        } else if (viewId == R.id.qr_pay) {
            changePayWay(binding.qrPay, PAY_QRCODE);

            // 关闭卡支付
            closeCardPaymentAccess();
            // 开启camera支付
            if (!hasCameraPermission) {
                ToastUtil.showToast(getActivity(), getString(R.string.txt_permission_failed));
            } else {
                Log.d(TAG, "isCameraOpened = " + isCameraOpened);
                if (!isCameraOpened) {
                    startSpot();
                }
            }
        } else if (viewId == R.id.btn_charge) {
            if (!binding.etCashReceived.getText().toString().trim().isEmpty()) {
                // 隱藏軟鍵盤
                hideSoftInput();
                // 更新界面狀態
                payCompleted(true);
            } else {
                ToastUtil.showToast(getActivity(), getString(R.string.txt_cash_received_empty));
            }
        } else if (viewId == R.id.btn_new_sale) {
            EventBus.getDefault().post(new ClearOrderEvent(true));
            ((MainTabActivity) getActivity()).hideChargeFragment();
        } else if (viewId == R.id.btn_retry) {
            removeCloseTimer();
            refreshPayPage(true);
            refreshPayTabState(binding.cashPay, PAY_CASH);
            curPayWay = PAY_CASH;
            hasPayResult = false;
        }
    }

    private void openCardPaymentAccess() {
        if (PaymentManager.getInstance(BaseApplicationV2.Companion.getContext()).isPaymentSdkAvailable()) {
            Log.d(TAG, "[openCardPaymentAccess]Begin to pay");
            PaymentManager.getInstance(BaseApplicationV2.Companion.getContext()).startToTrans(ticketTotalPrice, true, getActivity(), this::onPaymentSuccess);
        } else {
            ToastUtil.showToast(getActivity(), getString(R.string.txt_sdk_initializing));
        }
    }

    private void closeCardPaymentAccess() {
        if (PaymentManager.getInstance(BaseApplicationV2.Companion.getContext()).isPaymentSdkAvailable()) {
            PaymentManager.getInstance(BaseApplicationV2.Companion.getContext()).shutdownTrans();
        }
    }

    @Override
    public void onPaymentSuccess(boolean paySuccess, String msg, int resultCode) {
        Log.d(TAG, "onPaymentSuccess:: isSuccess = " + paySuccess + " , resultCode = " + resultCode);
        // 成功的时候显示支付成功界面
        // 失败的时候，除了“EMV_CANCEL=-1”外，都需要显示支付失败界面
        if (paySuccess || (!paySuccess && resultCode != -1)) {
            payCompleted(paySuccess);
        }
    }

    /**
     * 开启camera扫码
     */
    private void startSpot() {
        new Thread(() -> {
            Log.d(TAG, "startSpotAndShowRect+");
            binding.scanView.changeToScanQRCodeStyle(); // 切换成扫描二维码样式
            binding.scanView.setType(BarcodeType.TWO_DIMENSION, null); // 识别所有类型的二维码
            binding.scanView.startSpotAndShowRect(); // 显示扫描框，并开始识别
            isCameraOpened = true;
        }).start();
    }

    /**
     * 关闭camera扫码
     */
    private void stopSpot() {
        Log.d(TAG, "stopSpot+");
        binding.scanView.stopCamera();
        isCameraOpened = false;
    }

    @Override
    public void onScanQRCodeSuccess(String result) {
        Log.d(TAG, "onScanQRCodeSuccess");
        // 扫码成功
        payCompleted(true);
        // 播放支付成功提示音
        PaymentManager.getInstance(BaseApplicationV2.Companion.getContext()).playSuccessBeep();
    }

    @Override
    public void onCameraAmbientBrightnessChanged(boolean isDark) {

    }

    @Override
    public void onScanQRCodeOpenCameraError() {
        // 打开相机出错
        Log.e(TAG, "onScanQRCodeOpenCameraError");
    }

    private void changePayWay(CustomPayWay view, int payWay) {
        if (curPayWay != payWay) {
            refreshPayTabState(view, payWay);
            curPayWay = payWay;
        }
    }

    private void refreshPayTabState(CustomPayWay view, int payWay) {
        // 支付方式按键
        view.setPayWaySelected();
        // 支切换付方式布局
        binding.layoutReceived.setVisibility(payWay == PAY_CASH ? View.VISIBLE : View.GONE);
        binding.layoutCardPay.setVisibility(payWay == PAY_CARD ? View.VISIBLE : View.GONE);
        binding.layoutCameraPay.setVisibility(payWay == PAY_QRCODE ? View.VISIBLE : View.GONE);
    }

    private void refreshPayPage(boolean isShowPayWay) {
        binding.layoutPayWay.setVisibility(isShowPayWay ? View.VISIBLE : View.GONE);
        binding.layoutPayResult.setVisibility(isShowPayWay ? View.GONE : View.VISIBLE);
    }

    private void payCompleted(boolean isSuccess) {
        refreshPayPage(false);
        int payResultImg = isSuccess ? R.drawable.gif_pay_success : R.drawable.gif_pay_failed;
        Glide.with(this)
                .asGif()
                .load(payResultImg)
                .diskCacheStrategy(DiskCacheStrategy.RESOURCE)
                .into(binding.ivPayResult);
        binding.tvPayResult.setText(isSuccess ? getString(R.string.txt_pay_success) : getString(R.string.txt_pay_failed));
        binding.tvTotalPaidTip.setText(isSuccess ? getString(R.string.txt_total_paid) : getString(R.string.txt_total_amount_due));
        binding.tvPayVal.setText(requireContext().getString(R.string.money_sign) + ticketTotalPrice);
        binding.btnNewSale.setVisibility(isSuccess ? View.VISIBLE : View.GONE);
        binding.btnRetry.setVisibility(isSuccess ? View.GONE : View.VISIBLE);

        hasPayResult = true;

        // 10s后自动关闭该界面
        resetCloseTimer();
    }

    private void removeCloseTimer() {
        if (handler != null) {
            handler.removeCallbacks(closeRunnable);
        }
    }

    private void resetCloseTimer() {
        if (handler != null) {
            handler.removeCallbacks(closeRunnable);
            handler.postDelayed(closeRunnable, DELAY_CLOSE_PAGE_TIME);
        }
    }

    private void hideSoftInput() {
        View view = getActivity().getCurrentFocus();
        if (view != null) {
            InputMethodManager imm = (InputMethodManager) getActivity().getSystemService(Context.INPUT_METHOD_SERVICE);
            imm.hideSoftInputFromWindow(view.getWindowToken(), 0);
        }
    }

    private void release() {
        removeCloseTimer();
        stopSpot();
        closeCardPaymentAccess();
    }
}

