package com.xc.apex.nre.nrep8posdemo.view.fragment;

import android.app.AlertDialog;
import android.os.Bundle;
import android.os.Handler;
import android.os.Parcelable;
import android.util.Log;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.widget.TextView;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.databinding.DataBindingUtil;
import androidx.recyclerview.widget.GridLayoutManager;
import androidx.recyclerview.widget.LinearLayoutManager;
import androidx.recyclerview.widget.RecyclerView;

import com.xc.apex.nre.lib_payment.PaymentManager;
import com.xc.apex.nre.nrep8posdemo.BaseApplication;
import com.xc.apex.nre.nrep8posdemo.BaseApplicationV2;
import com.xc.apex.nre.nrep8posdemo.R;
import com.xc.apex.nre.nrep8posdemo.databinding.FragmentSalesBinding;
import com.xc.apex.nre.nrep8posdemo.model.ClearOrderEvent;
import com.xc.apex.nre.nrep8posdemo.model.GoodsBean;
import com.xc.apex.nre.nrep8posdemo.model.OrderItemBean;
import com.xc.apex.nre.nrep8posdemo.model.PayCompleted;
import com.xc.apex.nre.nrep8posdemo.utils.CommonUtils;
import com.xc.apex.nre.nrep8posdemo.utils.GoodsDataUtil;
import com.xc.apex.nre.nrep8posdemo.utils.ToastUtil;
import com.xc.apex.nre.nrep8posdemo.view.MainTabActivity;
import com.xc.apex.nre.nrep8posdemo.view.adapter.GridSpacingItemDecoration;
import com.xc.apex.nre.nrep8posdemo.view.adapter.SalesGoodsAdapter;
import com.xc.apex.nre.nrep8posdemo.view.adapter.SalesTicketAdapter;
import com.xc.apex.nre.nrep8posdemo.view.base.BaseFragment;

import org.greenrobot.eventbus.EventBus;
import org.greenrobot.eventbus.Subscribe;
import org.greenrobot.eventbus.ThreadMode;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.List;

public class SalesFragment extends BaseFragment implements SalesGoodsAdapter.OnItemClickListener,
        SalesTicketAdapter.OnItemClickListener, View.OnClickListener {

    // 单个商品最多添加数量
    private int maxGoods = 999;

    private FragmentSalesBinding binding;

    private List<GoodsBean> goodsDataList;
    private SalesGoodsAdapter salesGoodsAdapter;

    private List<OrderItemBean> orderItemList = new ArrayList<>();
    private SalesTicketAdapter salesTicketAdapter;

    private String ticketTotalPrice = "0.0";
    private boolean hasOrderNow = false;

    //定义一个主线名Hander
    private Handler mHandler = new Handler( );

    /**
     * 获取当前购物车内商品的数量
     * @return int
     */
    public int getOrderItemCount() {
        return orderItemList.size();
    }

    /**
     * 获取当前购物车商品列表
     */
    public List<OrderItemBean> getOrderItemList() {
        return orderItemList;
    }

    public FragmentSalesBinding getSalesBinding() {
        return binding;
    }

    @Nullable
    @Override
    public View onCreateView(@NonNull LayoutInflater inflater, @Nullable ViewGroup container, @Nullable Bundle savedInstanceState) {
        binding = DataBindingUtil.inflate(inflater, R.layout.fragment_sales, container, false);
        binding.setPresenter(this);

        EventBus.getDefault().register(this);

        // 商品列表
        goodsDataList = GoodsDataUtil.getInstance(getContext()).getGoodsListData();
        salesGoodsAdapter = new SalesGoodsAdapter(getContext(), goodsDataList);
        GridLayoutManager gridLayoutManager = new GridLayoutManager(getContext(), 3);
        binding.rvGoods.setLayoutManager(gridLayoutManager);
        binding.rvGoods.setAdapter(salesGoodsAdapter);
        int spacing = 12;
        boolean includeEdge = true;
        binding.rvGoods.addItemDecoration(new GridSpacingItemDecoration(gridLayoutManager.getSpanCount(), spacing, includeEdge));
        salesGoodsAdapter.setOnItemClickListener(this::onItemClick);

        // 订单列表
        salesTicketAdapter = new SalesTicketAdapter(orderItemList);
        RecyclerView.LayoutManager layoutManager = new LinearLayoutManager(getContext(), LinearLayoutManager.VERTICAL, false);
        binding.rvTicket.setLayoutManager(layoutManager);
        binding.rvTicket.setAdapter(salesTicketAdapter);
        salesTicketAdapter.setOnItemClickListener(this);

        if (CommonUtils.isSystemLanguageChinese(requireContext())) {
            binding.ivAllBar.setBackground(requireContext().getDrawable(R.drawable.ic_all_bar_zh));
        } else {
            binding.ivAllBar.setBackground(requireContext().getDrawable(R.drawable.ic_sales_all_bar));
        }

        return binding.getRoot();
    }

    @Override
    public void onDestroyView() {
        super.onDestroyView();
        EventBus.getDefault().unregister(this);
    }

    @Override
    public void onItemClick(View view, int position) {
        addOrderItem(position);
    }

    @Override
    public void onReduceClick(View view, int position, String unp) {
        subtractOrderItemSize(position, unp);
    }

    @Override
    public void onAddClick(View view, int position, String unp) {
        addOrderItemSize(position, unp);
    }

    @Override
    public void onClick(View view) {
        int viewId = view.getId();
        if (viewId == R.id.iv_ticket_clear) {
            if (hasOrderNow) {
                showClearDialog(true, getString(R.string.txt_clear_order));
            }
        } else if (viewId == R.id.tv_ticket_charge) {
            onOrderCreate();
        }
    }

    /**
     * 提交购物车并跳转到下单界面
     * EdgeOS协同需要用到下单，所以将这部分代码从 onClick 方法内提取出来复用
     */
    public void onOrderCreate() {
        mHandler.post(new Runnable() {
            @Override
            public void run() {
                if (!orderItemList.isEmpty()) {
                    if (!PaymentManager.getInstance(BaseApplicationV2.Companion.getContext()).isPaymentSdkAvailable()) {
                        showClearDialog(false, getString(R.string.txt_sdk_initializing));
                        return;
                    }

                    if (getActivity() != null) {
                        Bundle args = new Bundle();
                        args.putParcelableArrayList("order_data_key", (ArrayList<? extends Parcelable>) orderItemList);
                        args.putString("order_total_val_key", ticketTotalPrice);
                        ((MainTabActivity) getActivity()).showChargeFragment(args);
                    }
                }
            }
        });

    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onClearOrderEvent(ClearOrderEvent event) {
        if (event.isClear()) {
            clearAllOrder(false);
        }
    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onPayCompleted(PayCompleted event) {
        if (event.isSuccess()) {
            clearAllOrder(false);
            ToastUtil.showToast(getActivity(), getString(R.string.txt_pay_success));
        }
    }

    private void addOrderItem(int position) {
        addOrderItem(goodsDataList.get(position));
    }

    /**
     * 将商品添加到购物车中
     * 方法内会自动合并相同商品并自增计数
     * @param goods GoodsBean
     */
    public void addOrderItem(GoodsBean goods) {
        mHandler.post(new Runnable() {
            @Override
            public void run() {
                refreshOrderStateAndSendCommand(true);
                String goodsName = goods.getName();
                String unitPrice = goods.getUnitPrice();
                for (int i = 0; i < orderItemList.size(); i++) {
                    if (orderItemList.get(i).getId() == goods.getId()) {
                        addOrderItemSize(i, orderItemList.get(i).getUnitPrice());
                        return;
                    }
                }

                // 新增一条商品数据
                orderItemList.add(new OrderItemBean(goods.getId(), goodsName, 1, unitPrice, unitPrice));
                salesTicketAdapter.setData(orderItemList);
                // 更新订单总价
                addTickTotalVal(unitPrice);
            }
        });

    }

    private void addOrderItemSize(int position, String unitPrice) {
        OrderItemBean item = orderItemList.get(position);
        int size = item.getNum();
        if (size >= maxGoods) {
            return;
        }
        // 商品数量
        item.setNum(size + 1);
        // 商品总值
        item.setTotal(getAddPrice(unitPrice, item.getTotal()));
        // 刷新列表
        salesTicketAdapter.setData(orderItemList);
        // 更新订单总价
        addTickTotalVal(unitPrice);
    }

    private void addTickTotalVal(String unitP) {
        ticketTotalPrice = getAddPrice(unitP, ticketTotalPrice);
        binding.tvTicketTotal.setText(requireContext().getString(R.string.money_sign) + ticketTotalPrice);
    }

    private String getAddPrice(String unitP, String targetP) {
        BigDecimal unitPb = new BigDecimal(unitP);
        BigDecimal targetPb = new BigDecimal(targetP);
        return targetPb.add(unitPb).toString();
    }

    public void subtractOrderItemSize(GoodsBean goods) {
        mHandler.post(new Runnable() {
            @Override
            public void run() {
                // 从 orderItemList 查询所在位置，之后调用原方法
                for(int i = 0; i < orderItemList.size(); i++) {
                    if (orderItemList.get(i).getId() == goods.getId()) {
                        subtractOrderItemSize(i, orderItemList.get(i).getUnitPrice());
                        break;
                    }
                }
            }
        });

    }

    private void subtractOrderItemSize(int position, String unitPrice) {
        if (orderItemList.size() <= position) {
            return;
        }

        int size = orderItemList.get(position).getNum();
        if (size == 1) {
            // 刷新列表
            orderItemList.remove(position);
            salesTicketAdapter.setData(orderItemList);
            // 刷新订单总价
            subtractTickTotalVal(unitPrice);
            // 刷新按键状态
            if (orderItemList.size() == 0) {
                refreshOrderStateAndSendCommand(false);
            }
        } else {
            // 商品数量
            orderItemList.get(position).setNum(size - 1);
            // 商品总值
            orderItemList.get(position).setTotal(getSubtractPrice(unitPrice, orderItemList.get(position).getTotal()));
            salesTicketAdapter.setData(orderItemList);
            // 刷新订单总价
            subtractTickTotalVal(unitPrice);
        }
    }

    private void subtractTickTotalVal(String unitP) {
        ticketTotalPrice = getSubtractPrice(unitP, ticketTotalPrice);
        binding.tvTicketTotal.setText("$" + ticketTotalPrice);
    }

    private String getSubtractPrice(String unitP, String targetP) {
        BigDecimal unitPb = new BigDecimal(unitP);
        BigDecimal targetPb = new BigDecimal(targetP);
        return targetPb.subtract(unitPb).toString();
    }

    private void refreshOrderStateAndSendCommand(boolean hasOrder) {
        // 从后台线程切换到主线程更新 UI
        mHandler.post(new Runnable() {
            @Override
            public void run() {

                if (hasOrderNow != hasOrder) {
                    binding.tvTicketTotal.setTextColor(hasOrder ? getResources().getColor(R.color.black) : getResources().getColor(R.color.dark_98));
                    binding.ivTicketClear.setImageResource(hasOrder ? R.drawable.ic_clear_clickable : R.drawable.ic_clear_cannotclick);
                    binding.tvTicketCharge.setTextColor(hasOrder ? getResources().getColor(R.color.yellow)
                            : getResources().getColor(R.color.white));
                    binding.tvTicketCharge.setBackground(hasOrder ? getResources().getDrawable(R.drawable.bg_sales_charge_btn_clickable)
                            : getResources().getDrawable(R.drawable.bg_sales_charge_btn_cannotclick));
                    binding.ivEmpty.setVisibility(hasOrder ? View.GONE : View.VISIBLE);
                    binding.rvTicket.setVisibility(hasOrder ? View.VISIBLE : View.GONE);
                    hasOrderNow = hasOrder;
                }
            }
        });
    }

    private void refreshOrderState(boolean hasOrder) {
        if (hasOrderNow != hasOrder) {
            binding.tvTicketTotal.setTextColor(hasOrder ? getResources().getColor(R.color.black) : getResources().getColor(R.color.dark_98));
            binding.ivTicketClear.setImageResource(hasOrder ? R.drawable.ic_clear_clickable : R.drawable.ic_clear_cannotclick);
            binding.tvTicketCharge.setTextColor(hasOrder ? getResources().getColor(R.color.yellow)
                    : getResources().getColor(R.color.white));
            binding.tvTicketCharge.setBackground(hasOrder ? getResources().getDrawable(R.drawable.bg_sales_charge_btn_clickable)
                    : getResources().getDrawable(R.drawable.bg_sales_charge_btn_cannotclick));
            binding.ivEmpty.setVisibility(hasOrder ? View.GONE : View.VISIBLE);
            binding.rvTicket.setVisibility(hasOrder ? View.VISIBLE : View.GONE);
            hasOrderNow = hasOrder;
        }
    }

    private void showClearDialog(boolean isShowConfirmBtn, String titleStr) {
        LayoutInflater inflater = LayoutInflater.from(getActivity());
        View customView = inflater.inflate(R.layout.dialog_clear_ticket, null);

        AlertDialog.Builder builder = new AlertDialog.Builder(getActivity(), R.style.CustomDialogTheme);
        builder.setView(customView);
        AlertDialog alertDialog = builder.create();
        alertDialog.setCancelable(false);
        alertDialog.show();

        TextView title = customView.findViewById(R.id.tv_title);
        View vSeparate = customView.findViewById(R.id.v_separate);
        TextView cancel = customView.findViewById(R.id.cancel);
        TextView sure = customView.findViewById(R.id.sure);

        title.setText(titleStr);
        cancel.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View view) {
                alertDialog.dismiss();
            }
        });

        if (isShowConfirmBtn) {
            sure.setVisibility(View.VISIBLE);
            vSeparate.setVisibility(View.VISIBLE);
            sure.setOnClickListener(new View.OnClickListener() {
                @Override
                public void onClick(View view) {
                    clearAllOrder(true);
                    alertDialog.dismiss();
                }
            });
        } else {
            sure.setVisibility(View.GONE);
            vSeparate.setVisibility(View.GONE);
        }
    }

    public void clearAllOrder(boolean isNotifyCustomer) {
        mHandler.post(new Runnable() {
            @Override
            public void run() {
                orderItemList.clear();
                salesTicketAdapter.setData(orderItemList);
                if (isNotifyCustomer) {
                    refreshOrderStateAndSendCommand(false);
                } else {
                    refreshOrderState(false);
                }
                binding.tvTicketTotal.setText(getString(R.string.txt_def_total_value));
                ticketTotalPrice = "0.0";
            }
        });
    }
}
