package com.xc.apex.nre.nrep8posdemo.view.widget;

import android.animation.ValueAnimator;
import android.content.Context;
import android.graphics.Canvas;
import android.graphics.Color;
import android.graphics.Paint;
import android.graphics.RectF;
import android.util.AttributeSet;
import android.view.View;
import android.view.animation.LinearInterpolator;

import androidx.annotation.Nullable;

import java.util.ArrayList;
import java.util.List;
import java.util.Random;

public class VoiceWaveView extends View {

    private static final int BAR_COUNT = 15;
    private static final int BAR_WIDTH_DP = 6;
    private static final int BAR_GAP_DP = 6;
    private static final int CORNER_RADIUS_DP = 3;

    private Paint paint;
    private List<Float> barHeights;
    private List<ValueAnimator> animators;
    private Random random;
    private float barWidth;
    private float barGap;
    private float cornerRadius;
    private boolean isAnimating = false;

    public VoiceWaveView(Context context) {
        super(context);
        init();
    }

    public VoiceWaveView(Context context, @Nullable AttributeSet attrs) {
        super(context, attrs);
        init();
    }

    public VoiceWaveView(Context context, @Nullable AttributeSet attrs, int defStyleAttr) {
        super(context, attrs, defStyleAttr);
        init();
    }

    private void init() {
        float density = getContext().getResources().getDisplayMetrics().density;
        barWidth = BAR_WIDTH_DP * density;
        barGap = BAR_GAP_DP * density;
        cornerRadius = CORNER_RADIUS_DP * density;

        paint = new Paint(Paint.ANTI_ALIAS_FLAG);
        paint.setColor(Color.parseColor("#22D3EE")); // cyan-400
        paint.setStyle(Paint.Style.FILL);

        barHeights = new ArrayList<>();
        animators = new ArrayList<>();
        random = new Random();

        for (int i = 0; i < BAR_COUNT; i++) {
            barHeights.add(0.2f);
        }
    }

    public void startAnimation() {
        if (isAnimating) return;
        isAnimating = true;

        for (int i = 0; i < BAR_COUNT; i++) {
            final int index = i;
            ValueAnimator animator = ValueAnimator.ofFloat(0.2f, 1f, 0.2f);
            animator.setDuration(600 + random.nextInt(200));
            animator.setRepeatCount(ValueAnimator.INFINITE);
            animator.setRepeatMode(ValueAnimator.REVERSE);
            animator.setInterpolator(new LinearInterpolator());
            animator.setStartDelay(i * 50L);
            animator.addUpdateListener(animation -> {
                barHeights.set(index, (Float) animation.getAnimatedValue());
                invalidate();
            });
            animator.start();
            animators.add(animator);
        }
    }

    public void stopAnimation() {
        isAnimating = false;
        for (ValueAnimator animator : animators) {
            animator.cancel();
        }
        animators.clear();

        for (int i = 0; i < barHeights.size(); i++) {
            barHeights.set(i, 0.2f);
        }
        invalidate();
    }

    @Override
    protected void onMeasure(int widthMeasureSpec, int heightMeasureSpec) {
        int width = (int) (BAR_COUNT * barWidth + (BAR_COUNT - 1) * barGap);
        int height = MeasureSpec.getSize(heightMeasureSpec);
        setMeasuredDimension(width, height);
    }

    @Override
    protected void onDraw(Canvas canvas) {
        super.onDraw(canvas);

        int height = getHeight();
        float centerY = height / 2f;

        for (int i = 0; i < BAR_COUNT; i++) {
            float barHeight = height * barHeights.get(i);
            float left = i * (barWidth + barGap);
            float top = centerY - barHeight / 2;
            float right = left + barWidth;
            float bottom = centerY + barHeight / 2;

            RectF rect = new RectF(left, top, right, bottom);
            canvas.drawRoundRect(rect, cornerRadius, cornerRadius, paint);
        }
    }

    @Override
    protected void onDetachedFromWindow() {
        super.onDetachedFromWindow();
        stopAnimation();
    }
}